<?php
/**
 * Plugin Name: UpAmp Connector
 * Plugin URI: https://upamp.com/integrations/wordpress
 * Description: Connect your WordPress site to UpAmp for seamless blog publishing. Generate an API key and publish content directly from UpAmp without browser automation.
 * Version: 1.3.0
 * Author: UpAmp
 * Author URI: https://upamp.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: upamp-connector
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

define('UPAMP_CONNECTOR_VERSION', '1.3.0');
define('UPAMP_CONNECTOR_PLUGIN_DIR', plugin_dir_path(__FILE__));

class UpAmp_Connector {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
    }

    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_options_page(
            'UpAmp Connector',
            'UpAmp Connector',
            'manage_options',
            'upamp-connector',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('upamp_connector_settings', 'upamp_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('upamp_connector_settings', 'upamp_api_secret', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
    }

    /**
     * Enqueue admin styles and scripts
     */
    public function enqueue_admin_styles($hook) {
        if ('settings_page_upamp-connector' !== $hook) {
            return;
        }

        // Enqueue inline styles
        wp_add_inline_style('wp-admin', '
            .upamp-settings-wrap {
                max-width: 800px;
                margin-top: 20px;
            }
            .upamp-card {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin-bottom: 20px;
            }
            .upamp-card h2 {
                margin-top: 0;
                padding-bottom: 10px;
                border-bottom: 1px solid #eee;
            }
            .upamp-api-key-display {
                background: #f6f7f7;
                padding: 15px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 14px;
                word-break: break-all;
                margin: 10px 0;
            }
            .upamp-copy-btn {
                margin-left: 10px;
            }
            .upamp-endpoint {
                background: #e7f3ff;
                padding: 10px 15px;
                border-radius: 4px;
                font-family: monospace;
                margin: 10px 0;
            }
            .upamp-status {
                display: inline-block;
                padding: 5px 10px;
                border-radius: 3px;
                font-weight: 500;
            }
            .upamp-status-active {
                background: #d4edda;
                color: #155724;
            }
            .upamp-status-inactive {
                background: #f8d7da;
                color: #721c24;
            }
            .upamp-instructions {
                background: #fff8e5;
                border-left: 4px solid #ffb900;
                padding: 15px;
                margin: 15px 0;
            }
            .upamp-instructions ol {
                margin: 10px 0 0 20px;
            }
            .upamp-instructions li {
                margin-bottom: 8px;
            }
        ');

        // Register and enqueue inline script for clipboard functionality
        wp_register_script('upamp-admin', false, array(), UPAMP_CONNECTOR_VERSION, true);
        wp_enqueue_script('upamp-admin');

        $copied_message = esc_js(__('Copied to clipboard!', 'upamp-connector'));
        wp_add_inline_script('upamp-admin', "
            function copyToClipboard(elementId) {
                var text = document.getElementById(elementId).innerText;
                navigator.clipboard.writeText(text).then(function() {
                    alert('{$copied_message}');
                });
            }
        ");
    }

    /**
     * Generate a secure API key
     */
    private function generate_api_key() {
        return 'upamp_' . wp_generate_password(32, false);
    }

    /**
     * Generate a secure API secret
     */
    private function generate_api_secret() {
        return wp_generate_password(64, false);
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        // Handle key generation
        if (isset($_POST['upamp_generate_keys']) && check_admin_referer('upamp_generate_keys_nonce')) {
            $api_key = $this->generate_api_key();
            $api_secret = $this->generate_api_secret();
            update_option('upamp_api_key', $api_key);
            update_option('upamp_api_secret', wp_hash_password($api_secret));
            update_option('upamp_api_secret_display', $api_secret); // Store temporarily for display
            echo '<div class="notice notice-success"><p>' . esc_html__('New API credentials generated successfully!', 'upamp-connector') . '</p></div>';
        }

        // Handle key revocation
        if (isset($_POST['upamp_revoke_keys']) && check_admin_referer('upamp_revoke_keys_nonce')) {
            delete_option('upamp_api_key');
            delete_option('upamp_api_secret');
            delete_option('upamp_api_secret_display');
            echo '<div class="notice notice-warning"><p>' . esc_html__('API credentials have been revoked.', 'upamp-connector') . '</p></div>';
        }

        $api_key = get_option('upamp_api_key');
        $api_secret_display = get_option('upamp_api_secret_display');
        $site_url = get_rest_url(null, 'upamp/v1');
        $has_keys = !empty($api_key);

        ?>
        <div class="wrap upamp-settings-wrap">
            <h1><?php esc_html_e('UpAmp Connector', 'upamp-connector'); ?></h1>

            <div class="upamp-card">
                <h2><?php esc_html_e('Connection Status', 'upamp-connector'); ?></h2>
                <?php if ($has_keys): ?>
                    <p><span class="upamp-status upamp-status-active"><?php esc_html_e('Active', 'upamp-connector'); ?></span></p>
                    <p><?php esc_html_e('Your WordPress site is ready to receive content from UpAmp.', 'upamp-connector'); ?></p>
                <?php else: ?>
                    <p><span class="upamp-status upamp-status-inactive"><?php esc_html_e('Not Configured', 'upamp-connector'); ?></span></p>
                    <p><?php esc_html_e('Generate API credentials to connect your site to UpAmp.', 'upamp-connector'); ?></p>
                <?php endif; ?>
            </div>

            <div class="upamp-card">
                <h2><?php esc_html_e('API Credentials', 'upamp-connector'); ?></h2>

                <?php if ($has_keys): ?>
                    <p><strong><?php esc_html_e('API Key:', 'upamp-connector'); ?></strong></p>
                    <div class="upamp-api-key-display" id="api-key"><?php echo esc_html($api_key); ?></div>
                    <button type="button" class="button upamp-copy-btn" onclick="copyToClipboard('api-key')"><?php esc_html_e('Copy API Key', 'upamp-connector'); ?></button>

                    <?php if ($api_secret_display): ?>
                        <p style="margin-top: 20px;"><strong><?php esc_html_e('API Secret:', 'upamp-connector'); ?></strong></p>
                        <div class="upamp-api-key-display" id="api-secret"><?php echo esc_html($api_secret_display); ?></div>
                        <button type="button" class="button upamp-copy-btn" onclick="copyToClipboard('api-secret')"><?php esc_html_e('Copy API Secret', 'upamp-connector'); ?></button>
                        <p class="description" style="color: #d63638; margin-top: 10px;">
                            <strong><?php esc_html_e('Important:', 'upamp-connector'); ?></strong> <?php esc_html_e('Save this secret now! It will not be shown again after you leave this page.', 'upamp-connector'); ?>
                        </p>
                        <?php
                        // Clear the display secret after showing once
                        // delete_option('upamp_api_secret_display');
                        ?>
                    <?php else: ?>
                        <p style="margin-top: 20px;"><strong><?php esc_html_e('API Secret:', 'upamp-connector'); ?></strong> <em><?php esc_html_e('Hidden (regenerate to see new secret)', 'upamp-connector'); ?></em></p>
                    <?php endif; ?>

                    <p style="margin-top: 20px;"><strong><?php esc_html_e('API Endpoint:', 'upamp-connector'); ?></strong></p>
                    <div class="upamp-endpoint" id="api-endpoint"><?php echo esc_url($site_url); ?></div>
                    <button type="button" class="button upamp-copy-btn" onclick="copyToClipboard('api-endpoint')"><?php esc_html_e('Copy Endpoint', 'upamp-connector'); ?></button>

                    <hr style="margin: 20px 0;">

                    <form method="post" style="display: inline;">
                        <?php wp_nonce_field('upamp_generate_keys_nonce'); ?>
                        <button type="submit" name="upamp_generate_keys" class="button button-secondary"
                                onclick="return confirm('<?php echo esc_js(__('This will invalidate your current credentials. Continue?', 'upamp-connector')); ?>');">
                            <?php esc_html_e('Regenerate Credentials', 'upamp-connector'); ?>
                        </button>
                    </form>

                    <form method="post" style="display: inline; margin-left: 10px;">
                        <?php wp_nonce_field('upamp_revoke_keys_nonce'); ?>
                        <button type="submit" name="upamp_revoke_keys" class="button"
                                style="color: #d63638;"
                                onclick="return confirm('<?php echo esc_js(__('This will disconnect UpAmp from your site. Continue?', 'upamp-connector')); ?>');">
                            <?php esc_html_e('Revoke Credentials', 'upamp-connector'); ?>
                        </button>
                    </form>

                <?php else: ?>
                    <p><?php esc_html_e('No API credentials configured. Generate credentials to connect UpAmp.', 'upamp-connector'); ?></p>
                    <form method="post">
                        <?php wp_nonce_field('upamp_generate_keys_nonce'); ?>
                        <button type="submit" name="upamp_generate_keys" class="button button-primary">
                            <?php esc_html_e('Generate API Credentials', 'upamp-connector'); ?>
                        </button>
                    </form>
                <?php endif; ?>
            </div>

            <div class="upamp-card">
                <h2><?php esc_html_e('Setup Instructions', 'upamp-connector'); ?></h2>
                <div class="upamp-instructions">
                    <ol>
                        <li><?php esc_html_e('Click "Generate API Credentials" above to create your API key and secret.', 'upamp-connector'); ?></li>
                        <li><?php echo wp_kses(__('Copy the <strong>API Endpoint</strong>, <strong>API Key</strong>, and <strong>API Secret</strong>.', 'upamp-connector'), array('strong' => array())); ?></li>
                        <li><?php esc_html_e('In UpAmp, go to Settings → WordPress Integration.', 'upamp-connector'); ?></li>
                        <li><?php esc_html_e('Select "UpAmp Plugin" as the connection method.', 'upamp-connector'); ?></li>
                        <li><?php esc_html_e('Paste your API Endpoint, Key, and Secret.', 'upamp-connector'); ?></li>
                        <li><?php esc_html_e('Click "Test Connection" to verify everything works.', 'upamp-connector'); ?></li>
                    </ol>
                </div>
            </div>

            <div class="upamp-card">
                <h2><?php esc_html_e('Available Endpoints', 'upamp-connector'); ?></h2>
                <table class="widefat" style="margin-top: 10px;">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Endpoint', 'upamp-connector'); ?></th>
                            <th><?php esc_html_e('Method', 'upamp-connector'); ?></th>
                            <th><?php esc_html_e('Description', 'upamp-connector'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>/upamp/v1/publish</code></td>
                            <td>POST</td>
                            <td><?php esc_html_e('Create a new blog post', 'upamp-connector'); ?></td>
                        </tr>
                        <tr>
                            <td><code>/upamp/v1/test</code></td>
                            <td>GET</td>
                            <td><?php esc_html_e('Test connection and authentication', 'upamp-connector'); ?></td>
                        </tr>
                        <tr>
                            <td><code>/upamp/v1/categories</code></td>
                            <td>GET</td>
                            <td><?php esc_html_e('List available categories', 'upamp-connector'); ?></td>
                        </tr>
                        <tr>
                            <td><code>/upamp/v1/tags</code></td>
                            <td>GET</td>
                            <td><?php esc_html_e('List available tags', 'upamp-connector'); ?></td>
                        </tr>
                        <tr>
                            <td><code>/upamp/v1/update/{post_id}</code></td>
                            <td>PUT</td>
                            <td><?php esc_html_e('Update an existing blog post', 'upamp-connector'); ?></td>
                        </tr>
                        <tr>
                            <td><code>/upamp/v1/delete/{post_id}</code></td>
                            <td>DELETE</td>
                            <td><?php esc_html_e('Delete a blog post (moves to trash)', 'upamp-connector'); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Test endpoint
        register_rest_route('upamp/v1', '/test', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_test'),
            'permission_callback' => array($this, 'authenticate_request'),
        ));

        // Publish endpoint
        register_rest_route('upamp/v1', '/publish', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_publish'),
            'permission_callback' => array($this, 'authenticate_request'),
        ));

        // Categories endpoint
        register_rest_route('upamp/v1', '/categories', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_get_categories'),
            'permission_callback' => array($this, 'authenticate_request'),
        ));

        // Tags endpoint
        register_rest_route('upamp/v1', '/tags', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_get_tags'),
            'permission_callback' => array($this, 'authenticate_request'),
        ));

        // Update endpoint
        register_rest_route('upamp/v1', '/update/(?P<post_id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array($this, 'handle_update'),
            'permission_callback' => array($this, 'authenticate_request'),
            'args' => array(
                'post_id' => array(
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Delete endpoint
        register_rest_route('upamp/v1', '/delete/(?P<post_id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array($this, 'handle_delete'),
            'permission_callback' => array($this, 'authenticate_request'),
            'args' => array(
                'post_id' => array(
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));
    }

    /**
     * Authenticate API request
     */
    public function authenticate_request(WP_REST_Request $request) {
        $api_key = get_option('upamp_api_key');
        $api_secret_hash = get_option('upamp_api_secret');

        if (empty($api_key) || empty($api_secret_hash)) {
            return new WP_Error('not_configured', 'UpAmp Connector is not configured', array('status' => 401));
        }

        // Get credentials from header
        $auth_header = $request->get_header('X-UpAmp-Auth');

        if (empty($auth_header)) {
            // Try Authorization header as fallback
            $auth_header = $request->get_header('Authorization');
            if (!empty($auth_header) && strpos($auth_header, 'Bearer ') === 0) {
                $auth_header = substr($auth_header, 7);
            }
        }

        if (empty($auth_header)) {
            return new WP_Error('missing_auth', 'Missing authentication header', array('status' => 401));
        }

        // Parse key:secret format
        $parts = explode(':', $auth_header, 2);
        if (count($parts) !== 2) {
            return new WP_Error('invalid_auth', 'Invalid authentication format', array('status' => 401));
        }

        $provided_key = $parts[0];
        $provided_secret = $parts[1];

        // Verify key
        if ($provided_key !== $api_key) {
            return new WP_Error('invalid_key', 'Invalid API key', array('status' => 401));
        }

        // Verify secret
        if (!wp_check_password($provided_secret, $api_secret_hash)) {
            return new WP_Error('invalid_secret', 'Invalid API secret', array('status' => 401));
        }

        return true;
    }

    /**
     * Handle test endpoint
     */
    public function handle_test(WP_REST_Request $request) {
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'UpAmp Connector is working',
            'site_name' => get_bloginfo('name'),
            'site_url' => get_site_url(),
            'wordpress_version' => get_bloginfo('version'),
            'plugin_version' => UPAMP_CONNECTOR_VERSION,
        ), 200);
    }

    /**
     * Handle publish endpoint
     */
    public function handle_publish(WP_REST_Request $request) {
        $params = $request->get_json_params();

        // Validate required fields
        if (empty($params['title'])) {
            return new WP_Error('missing_title', 'Post title is required', array('status' => 400));
        }

        if (empty($params['content'])) {
            return new WP_Error('missing_content', 'Post content is required', array('status' => 400));
        }

        // Prepare post data
        $post_data = array(
            'post_title' => sanitize_text_field($params['title']),
            'post_content' => wp_kses_post($params['content']),
            'post_status' => isset($params['status']) ? sanitize_text_field($params['status']) : 'draft',
            'post_author' => $this->get_default_author(),
            'post_type' => 'post',
        );

        // Add excerpt if provided
        if (!empty($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }

        // Handle categories
        if (!empty($params['categories']) && is_array($params['categories'])) {
            $category_ids = $this->process_categories($params['categories']);
            if (!empty($category_ids)) {
                $post_data['post_category'] = $category_ids;
            }
        }

        // Create the post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return new WP_Error('create_failed', $post_id->get_error_message(), array('status' => 500));
        }

        // Handle tags
        if (!empty($params['tags']) && is_array($params['tags'])) {
            $this->process_tags($post_id, $params['tags']);
        }

        // Handle featured image
        if (!empty($params['featuredImage'])) {
            $this->set_featured_image($post_id, $params['featuredImage']);
        }

        // Handle meta description (if Yoast or similar is installed)
        if (!empty($params['metaDescription'])) {
            $this->set_meta_description($post_id, $params['metaDescription']);
        }

        // Get the post URL
        $post_url = get_permalink($post_id);

        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'Post created successfully',
            'post_id' => $post_id,
            'post_url' => $post_url,
            'edit_url' => get_edit_post_link($post_id, 'raw'),
        ), 201);
    }

    /**
     * Handle update endpoint - update an existing post
     */
    public function handle_update(WP_REST_Request $request) {
        $post_id = intval($request->get_param('post_id'));
        $params = $request->get_json_params();

        // Verify the post exists
        $existing_post = get_post($post_id);
        if (!$existing_post) {
            return new WP_Error('post_not_found', 'Post not found', array('status' => 404));
        }

        // Verify it's a post type we can update
        if ($existing_post->post_type !== 'post') {
            return new WP_Error('invalid_post_type', 'Can only update posts', array('status' => 400));
        }

        // Prepare update data
        $post_data = array(
            'ID' => $post_id,
        );

        // Update title if provided
        if (!empty($params['title'])) {
            $post_data['post_title'] = sanitize_text_field($params['title']);
        }

        // Update content if provided
        if (!empty($params['content'])) {
            $post_data['post_content'] = wp_kses_post($params['content']);
        }

        // Update excerpt if provided
        if (isset($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }

        // Update status if provided
        if (!empty($params['status'])) {
            $post_data['post_status'] = sanitize_text_field($params['status']);
        }

        // Update the post
        $result = wp_update_post($post_data, true);

        if (is_wp_error($result)) {
            return new WP_Error('update_failed', $result->get_error_message(), array('status' => 500));
        }

        // Handle categories update
        if (isset($params['categories']) && is_array($params['categories'])) {
            $category_ids = $this->process_categories($params['categories']);
            wp_set_post_categories($post_id, $category_ids);
        }

        // Handle tags update
        if (isset($params['tags']) && is_array($params['tags'])) {
            $this->process_tags($post_id, $params['tags']);
        }

        // Handle featured image update
        if (!empty($params['featuredImage'])) {
            // Remove old featured image first
            $old_thumbnail_id = get_post_thumbnail_id($post_id);
            if ($old_thumbnail_id) {
                delete_post_thumbnail($post_id);
            }
            $this->set_featured_image($post_id, $params['featuredImage']);
        }

        // Handle meta description update
        if (!empty($params['metaDescription'])) {
            $this->set_meta_description($post_id, $params['metaDescription']);
        }

        // Get the updated post URL
        $post_url = get_permalink($post_id);

        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'Post updated successfully',
            'post_id' => $post_id,
            'post_url' => $post_url,
            'edit_url' => get_edit_post_link($post_id, 'raw'),
        ), 200);
    }

    /**
     * Handle delete endpoint - delete (trash) an existing post
     */
    public function handle_delete(WP_REST_Request $request) {
        $post_id = intval($request->get_param('post_id'));
        $params = $request->get_json_params();
        $force_delete = isset($params['force']) && $params['force'] === true;

        // Verify the post exists
        $existing_post = get_post($post_id);
        if (!$existing_post) {
            return new WP_Error('post_not_found', 'Post not found', array('status' => 404));
        }

        // Verify it's a post type we can delete
        if ($existing_post->post_type !== 'post') {
            return new WP_Error('invalid_post_type', 'Can only delete posts', array('status' => 400));
        }

        // Delete the post (moves to trash by default, or permanently if force=true)
        if ($force_delete) {
            $result = wp_delete_post($post_id, true);
        } else {
            $result = wp_trash_post($post_id);
        }

        if (!$result) {
            return new WP_Error('delete_failed', 'Failed to delete post', array('status' => 500));
        }

        return new WP_REST_Response(array(
            'success' => true,
            'message' => $force_delete ? 'Post permanently deleted' : 'Post moved to trash',
            'post_id' => $post_id,
        ), 200);
    }

    /**
     * Get default author ID
     */
    private function get_default_author() {
        // Get the first administrator
        $admins = get_users(array('role' => 'administrator', 'number' => 1));
        if (!empty($admins)) {
            return $admins[0]->ID;
        }
        return 1;
    }

    /**
     * Process categories - find existing or create new
     */
    private function process_categories($categories) {
        $category_ids = array();

        foreach ($categories as $category_name) {
            $category_name = sanitize_text_field($category_name);

            // Try to find existing category
            $term = get_term_by('name', $category_name, 'category');

            if ($term) {
                $category_ids[] = $term->term_id;
            } else {
                // Create new category
                $new_term = wp_insert_term($category_name, 'category');
                if (!is_wp_error($new_term)) {
                    $category_ids[] = $new_term['term_id'];
                }
            }
        }

        return $category_ids;
    }

    /**
     * Process tags - find existing or create new
     */
    private function process_tags($post_id, $tags) {
        $tag_names = array_map('sanitize_text_field', $tags);
        wp_set_post_tags($post_id, $tag_names, false);
    }

    /**
     * Set featured image from URL
     */
    private function set_featured_image($post_id, $image_url) {
        // Include required files
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download the image
        $tmp = download_url($image_url);

        if (is_wp_error($tmp)) {
            return false;
        }

        // Get file info
        $file_array = array(
            'name' => basename(wp_parse_url($image_url, PHP_URL_PATH)),
            'tmp_name' => $tmp,
        );

        // If no extension, try to detect from content type
        if (!preg_match('/\.(jpg|jpeg|png|gif|webp)$/i', $file_array['name'])) {
            $file_array['name'] .= '.jpg';
        }

        // Upload the image
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temp file
        if (file_exists($tmp)) {
            wp_delete_file($tmp);
        }

        if (is_wp_error($attachment_id)) {
            return false;
        }

        // Set as featured image
        set_post_thumbnail($post_id, $attachment_id);

        return $attachment_id;
    }

    /**
     * Set meta description (supports Yoast SEO, Rank Math, All in One SEO)
     */
    private function set_meta_description($post_id, $description) {
        $description = sanitize_textarea_field($description);

        // Yoast SEO
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
        }

        // Rank Math
        if (class_exists('RankMath')) {
            update_post_meta($post_id, 'rank_math_description', $description);
        }

        // All in One SEO
        if (class_exists('AIOSEOP_Core')) {
            update_post_meta($post_id, '_aioseop_description', $description);
        }

        // Store as generic meta as well
        update_post_meta($post_id, '_upamp_meta_description', $description);
    }

    /**
     * Handle get categories endpoint
     */
    public function handle_get_categories(WP_REST_Request $request) {
        $categories = get_categories(array(
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        ));

        $result = array();
        foreach ($categories as $category) {
            $result[] = array(
                'id' => $category->term_id,
                'name' => $category->name,
                'slug' => $category->slug,
                'count' => $category->count,
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'categories' => $result,
        ), 200);
    }

    /**
     * Handle get tags endpoint
     */
    public function handle_get_tags(WP_REST_Request $request) {
        $tags = get_tags(array(
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        ));

        $result = array();
        foreach ($tags as $tag) {
            $result[] = array(
                'id' => $tag->term_id,
                'name' => $tag->name,
                'slug' => $tag->slug,
                'count' => $tag->count,
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'tags' => $result,
        ), 200);
    }
}

// Initialize the plugin
add_action('plugins_loaded', array('UpAmp_Connector', 'get_instance'));
